/******************************************************************************
 *
 *	Lee Haywood's SokEvo - a Sokoban puzzle generator.
 *	Copyright (C) 2009.
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version 2
 *	of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to:
 *
 *		The Free Software Foundation, Inc.,
 *		59 Temple Place - Suite 330, Boston, MA
 *		02111-1307, USA.
 *
 ******************************************************************************
 *
 * editor.js : This JavaScript program allows templates held on the server to
 *	       be edited in a web browser.  The data is read and written
 *	       asynchronously.
 *
 *****************************************************************************/





// *** INITIALISATION PROCESSING ***

// Find the status box, reset associated values.
globalStatusNode = document.getElementById( "Status" )
globalStatusRed = 0
globalStatusGreen = 0
globalStatusBlue = 0
globalStatusTimer = null

// Find the warning section.
globalWarningNode = document.getElementById( "Warning" )

// Find the template list section.
globalList = document.getElementById( "List" )
globalListCopy = document.getElementById( "List2" )

// Find the row and column addition tables.
globalRowAdder = document.getElementById( "RowAdder" )
globalColumnAdder = document.getElementById( "ColumnAdder" )
globalColumnRemover = document.getElementById( "ColumnRemover" )

// Find the main template table.
globalDataTable = document.getElementById( "Template" )
globalDisplayHeight = 0
globalDisplayWidth = 0

// Find the undo and redo links/images and hide them initially.
globalUndo = document.getElementById( "UNDO" )
globalRedo = document.getElementById( "REDO" )
globalUndo.style.visibility = "hidden"
globalRedo.style.visibility = "hidden"

// Start with no template.
globalActiveTemplate = -1
globalNextTemplate = -1
globalTemplate = ""
globalInitialTemplate = null
globalInitialHeight = 0
globalInitialWidth = 0

// Prepare to store undo/redo history.
globalActionList = new Array
globalActionNum = 0
globalNumActions = 0
globalReplay = false

// Maximum length of a template title.
globalTitleSize = 30

// Size of cell images.
globalConfigCellSize = 50

// Find spacing control and update it with value from cookie or default size.
globalCellSize = readCookie( "CELLSIZE" )
if ( globalCellSize == null || ! globalCellSize.match( "^[1-9][0-9]?$" ) ||
     parseInt( globalCellSize, 10 ) < globalConfigCellSize ||
     parseInt( globalCellSize, 10 ) > globalConfigCellSize * 2 )
{
    globalCellSize = globalConfigCellSize + 4
}
else
{
    globalCellSize = parseInt( globalCellSize, 10 )
}
globalCellSize -= globalCellSize % 2

// Set the active cell type.
globalCellType = null
setType( '#' )

// Get object required to connect to server - try browser-specific type first.
globalServerObject = null
try
{
    globalServerObject = new ActiveXObject( "Msxml2.XMLHTTP" )
}
catch ( objectException )
{
    try
    {
	globalServerObject = new ActiveXObject( "Microsoft.XMLHTTP" )
    }
    catch ( objectException2 )
    {
	globalServerObject = null
    }
}

// Default to generic request object if not found.
if ( globalServerObject == null )
{
    if ( typeof XMLHttpRequest != "undefined" )
    {
	globalServerObject = new XMLHttpRequest()
    }
}

// Start with an empty server request queue.
globalRequestQueue = new Array
globalQueueSize = 0

// Request the list of templates from the server asynchronously.  The first
// template will be loaded automatically once its title has been received.
globalTemplateList = null
serverRequest( "Load list", "GET", "/?Form=Templates" )

// Create a dummy row/cell at the top of the template list, to push the
// first entry down towards the top of the main editor table.
var newRow = globalList.insertRow( 0 )
var newCell = newRow.insertCell( 0 )
newCell.setAttribute( "height", 2 * globalConfigCellSize )



// Adds an entry to the server request queue and, if the queue was empty, kicks
// off the process of sending that request to the server.  All requests (both
// loads and saves) are performed asynchronously, and use of this entry point
// ensures that they all take place in the correct sequence (one at a time).

function serverRequest( requestType, requestMethod, newRequest )
{
    // Add the new message to the queue, simplify encoding of any spaces.
    globalRequestQueue[ globalQueueSize++ ] =
			    new queueInfo( requestType, requestMethod,
					   newRequest.replace( /%20/g, "+" ) )

    // If this is the only entry in the queue, send it now.
    if ( globalQueueSize == 1 )
    {
	sendServerRequest()
    }
}



// Sends the first entry in the request queue to the server and removes it.

function sendServerRequest()
{
    // Store the request, change the server mode to match.
    globalRequestType = globalRequestQueue[ 0 ].requestType
    var requestMethod = globalRequestQueue[ 0 ].requestMethod
    var requestData = globalRequestQueue[ 0 ].requestData

    // Remove the request from the queue.
    for ( var index = 1; index < globalQueueSize; index++ )
    {
	globalRequestQueue[ index - 1 ] = globalRequestQueue[ index ]
    }
    globalQueueSize--

    // Show fixed status, based on the request type.
    if ( globalRequestType.substring( 0, 4 ) == "Load" )
    {
	showStatus( "Loading...", 0, 128, 0, 0 )
    }
    else
    {
	showStatus( "Saving...", 255, 0, 0, 0 )
    }

    // Start sending the first action in the queue to the server.
    if ( requestMethod == "POST" )
    {
	globalServerObject.open( requestMethod, "/" + getUniqueID(), true )
	globalServerObject.onreadystatechange = receiveServerResponse
	globalServerObject.send( requestData )
    }
    else
    {
	globalServerObject.open( requestMethod,
				 requestData + "&" + getUniqueID(), true )
	globalServerObject.onreadystatechange = receiveServerResponse
	globalServerObject.send( null )
    }
}



// Handles the response from the server after data has been loaded from it.

function receiveServerResponse()
{
    var actionValue

    // Ensure that the response has been fully received from the server.
    if ( globalServerObject.readyState != 4 )
    {
	return
    }

    // Handle any server error that occurred.
    // Note: Microsoft mangle status 204 (no content) as 1223.
    if ( globalServerObject.status != 200 &&
	 ( ( globalRequestType != "Rename" && globalRequestType != "Save" ) ||
	   ( globalServerObject.status != 204 &&
	     globalServerObject.status != 1223 ) ) )
    {
	showError( globalServerObject.status + " - " +
		   globalServerObject.statusText, true )
	document.getElementById( "ServerPage" ).innerHTML =
					    globalServerObject.responseText
	return
    }

    // If the list of templates is expected...
    if ( globalRequestType == "Load list" || globalRequestType == "Delete" )
    {
	// Store the list of templates.
	var rawData = globalServerObject.responseText
	globalTemplateList = rawData.split( '&' )
    }
    // Otherwise, if a specific template was requested...
    else if ( globalRequestType == "Load template" )
    {
	var templateData = globalServerObject.responseXML
	var rootNode = templateData.getElementsByTagName( 'template' ).item( 0 )

	// Make the template active.
	globalActiveTemplate = globalNextTemplate

	// Remove the existing list of links for templates.
	while ( globalList.rows.length > 2 )
	{
	    globalList.deleteRow( globalList.rows.length - 1 )
	    globalListCopy.deleteRow( globalListCopy.rows.length - 1 )
	}

	// List the names of each template name received, with links.
	for ( var index = 0; index < globalTemplateList.length; index++ )
	{
	    addTemplate( index, false )
	}

	// Store the template and its size.
	globalHeight = 2 + parseInt( findElement( templateData, 'height' ), 10 )
	globalWidth = 2 + parseInt( findElement( templateData, 'width' ), 10 )
	globalTemplate =
		findElement( templateData, 'definition' ).replace( /_/g, " " )

	// Store the undo/redo history.
	globalActionList = new Array
	globalActionNum = 0
	globalNumActions = 0
	var numActions =
		    parseInt( findElement( templateData, 'numactions' ), 10 )
	var actionList = decodeURIComponent(
			findElement( templateData, 'history' ) ).split( "&" )
	for ( var index = 1; index < numActions; index++ )
	{
	    actionValue = actionList[ index ]
	    if ( actionValue.substring( 0, 1 ) == '[' )
	    {
		storeAction( "=",
		     actionValue.substring( 1, 2 ) + "&" +
		     actionValue.substring( 4 ).replace( / /g, "&" ), false )
	    }
	    else
	    {
		storeAction( actionValue.substring( 0, 2 ),
			     actionValue.substring( 2 ), false )
	    }
	}
	globalActionNum =
		parseInt( findElement( templateData, 'actionnum' ), 10 ) - 1

	// Update the visibility of the undo/redo images/links.
	setReplayState()

	// Generate the initial template, to facilitate undo/redo operations.
	var sizeData = actionList[ 0 ].split( ' ' )
	globalInitialHeight = parseInt( sizeData[ 0 ], 10 ) + 2
	globalInitialWidth = parseInt( sizeData[ 1 ], 10 ) + 2
	globalInitialTemplate = initialiseTemplate( globalInitialHeight,
						    globalInitialWidth )

	// Display the template.
	displayTable()
    }

    // If the template list was just received, display it and then load the
    // most recently edited template (making it active)...
    if ( globalRequestType == "Load list" || globalRequestType == "Delete" )
    {
	goToTemplate( 0 )
    }
    // Otherwise, if a new template was requested...
    else if ( globalRequestType == "New template" )
    {
	// Add the title received from the server to start of the template list.
	var newName = globalServerObject.responseText
	globalTemplateList = new Array( newName ).concat( globalTemplateList )

	// Display the new entry in the template list.
	goToTemplate( 0 )
    }
    // Otherwise, if the queue has become empty, update the status display...
    else if ( globalQueueSize <= 0 )
    {
	if ( globalRequestType.substring( 0, 4 ) == "Load" )
	{
	    showStatus( "Loaded", 0, 128, 0, 1 )
	}
	else
	{
	    showStatus( "Saved", 0, 128, 0, 1 )
	}
	globalRequestType = null
    }
    // Otherwise, start sending next action in the request queue to the server.
    else
    {
	sendServerRequest()
    }
}



// Searches from the given XML root node to find the element with the specified
// name, returning its value as a string.

function findElement( rootNode, itemName )
{
    // Return an empty string if the node is missing (it may just be empty).
    var xmlNode = rootNode.getElementsByTagName( itemName ).item( 0 ).firstChild
    if ( xmlNode == null )
    {
	return ""
    }

    // Otherwise, return the value from the node as a string.
    return "" + xmlNode.nodeValue
}



// Switches context to the specified template.

function goToTemplate( templateNum )
{
    // Ignore the request if already loading.
    if ( globalRequestType == "Load template" )
    {
	return
    }

    // Store the number of the template being loaded.
    globalNextTemplate = templateNum

    // Request the template data from the server asynchronously.
    serverRequest( "Load template", "GET", "/?Form=Template&Title=" +
					   globalTemplateList[ templateNum ] )
}



// Initialises a template with a given height and width.  This is used for
// existing templates, where the current state is reached from an initially
// empty grid by 'replaying' the actions in the undo/redo history.

function initialiseTemplate( numRows, numColumns )
{
    var index

    // Create lines for top/bottom walls and inner spaces.
    var wall = "#"
    var space = "#"
    for ( index = 2; index < numColumns; index++ )
    {
	wall += "#"
	space += " "
    }
    space += "#"
    wall += "#"

    // Start with upper wall, add inner rows of spaces.
    var result = wall
    for ( index = 2; index < numRows; index++ )
    {
	result += space
    }

    // Return final result, adding bottom wall.
    return result + wall
}



// Displays the template, with links to allow the user to add/remove rows and
// columns as well as change the individual cells.  If the number of rows or
// columns is changed, this can be re-used to simply destroy and re-create the
// table.

function displayTable()
{
    var editRow, newRow, newCell
    var functionName
    var rowNum, columnNum
    var cellIndex, cellType, cellColour

    // Do nothing if the display has been deactivated for an undo/redo.
    if ( globalReplay )
    {
	return
    }

    // Display number of rows and columns.
    document.getElementById( "Title" ).innerHTML =
		"<P><TT>" + ( globalHeight - 2 ) + "&#215;" +
			    ( globalWidth - 2 ) + "</TT></P>"

    // Make the main template table temporarily invisible, to reduce flicker.
    globalDataTable.style.visibility = "hidden"

    // Discard all existing rows/cells in each table, ready to re-create them.
    // Only re-create row/column editors if height/width has changed.
    while ( globalDataTable.rows.length > 0 )
    {
	globalDataTable.deleteRow( globalDataTable.rows.length - 1 )
    }
    if ( globalHeight != globalDisplayHeight )
    {
	while ( globalRowAdder.rows.length > 0 )
	{
	    globalRowAdder.deleteRow( globalRowAdder.rows.length - 1 )
	}
    }

    // If the height has changed, create a dummy row/cell in the row editor
    // table, to push the first addition link/image down.
    if ( globalHeight != globalDisplayHeight )
    {
	editRow = globalRowAdder.insertRow( 0 )
	newCell = editRow.insertCell( 0 )
	newCell.setAttribute( "height", globalCellSize - 9 )
    }

    // For each row...
    cellIndex = 0
    for ( rowNum = 0; rowNum < globalHeight; rowNum++ )
    {
	// If beneath the top of the outer wall...
	if ( rowNum > 0 )
	{
	    // If the height has changed...
	    if ( globalHeight != globalDisplayHeight )
	    {
		// Create a corresponding row in the row editor table.
		editRow = globalRowAdder.insertRow( rowNum )
		editRow.setAttribute( "valign", "top" )

		// Add link/image to add a row to the row editor table.
		addImageLink( editRow, 0, "addRow", rowNum,
			      "Insert row", "add.gif", null,
			  ( rowNum < globalHeight - 1 ) ? globalCellSize : 0, 0,
			      null, "top" )
	    }
	}

	// Create a row for the for the template itself.
	newRow = globalDataTable.insertRow( rowNum )

	// For each column...
	for ( columnNum = 0; columnNum < globalWidth; columnNum++ )
	{
	    // If the cell may be edited, add link to it and highlight it if
	    // it is locked.
	    if ( rowNum > 0 && rowNum < globalHeight - 1 &&
		 columnNum > 0 && columnNum < globalWidth - 1 )
	    {
		functionName = "setCell"
		cellType = globalTemplate.substring( cellIndex, cellIndex + 1 )
		if ( cellType == '^' || cellType == ';' ||  cellType == '%' )
		{
		    cellColour = "#FF0000"
		}
		else
		{
		    cellColour = "#FFFFFF"
		}
	    }
	    else
	    {
		functionName = null
		cellColour = null
	    }

	    // Add a cell to display the image.
	    addImageLink( newRow, columnNum, functionName,
			  rowNum + "," + columnNum, null,
			  getImageName( rowNum, columnNum ), cellColour,
			  globalCellSize, globalCellSize,
//			  globalConfigCellSize, globalConfigCellSize,
			  "center", "middle" )

	    // Advance to the next cell.
	    cellIndex++
	}

	// Add final column for delete image, with link to delete the row.
	// Do not provide any delete options if the table is already too small.
	if ( rowNum > 0 && rowNum < globalHeight - 1 && globalHeight > 4 )
	{
	    addImageLink( newRow, globalWidth, "deleteRow", rowNum,
		       "Delete row", "delete.gif", null, 0, 0, null, "middle" )
	}
    }

    // If the table width has changed, or become invalid...
    if ( globalWidth != globalDisplayWidth )
    {
	// Remove existing column addition links/images.
	while ( globalColumnAdder.firstChild != null )
	{
	    globalColumnAdder.removeChild( globalColumnAdder.firstChild )
	}

	// Remove existing column deletion links/images.
	while ( globalColumnRemover.firstChild != null )
	{
	    globalColumnRemover.removeChild( globalColumnRemover.firstChild )
	}

	// Create a dummy cell in the column editor table, to push the first
	// addition link/image to the right.
	newCell = globalColumnAdder.insertCell( 0 )
	newCell.setAttribute( "width", globalCellSize - 8 )

	// Create a dummy cell in the column deletion table, to push the first
	// deletion link/image to the right.
	newCell = globalColumnRemover.insertCell( 0 )
	newCell.setAttribute( "width", globalCellSize )

	// For each column...
	for ( columnNum = 1; columnNum < globalWidth; columnNum++ )
	{
	    // Add image in column editor table, with link to add a column.
	    if ( globalWidth != globalDisplayWidth )
	    {
		addImageLink( globalColumnAdder, columnNum, "addColumn",
			      columnNum, "Insert column", "add.gif", null, 0,
			      ( columnNum < globalWidth ) ? globalCellSize : 0,
			      "left", null )
	    }

	    // Add delete image to template table, with link to delete column.
	    // Do not provide any delete options if table is already too small.
	    if ( columnNum < globalWidth - 1 && globalWidth > 4 )
	    {
		addImageLink( globalColumnRemover, columnNum,
			      "deleteColumn", columnNum,
			      "Delete column", "delete.gif", null,
			      0, globalCellSize, "center", null )
	    }
	}
    }

    // Make the main template table visible, remember displayed height/width.
    globalDataTable.style.visibility = "visible"
    globalDisplayHeight = globalHeight
    globalDisplayWidth = globalWidth
}



// Creates a table cell, with a linked-image that calls the specified function
// with the given value when the user selects the named image.  Optionally, a
// description, height, width and vertical/horizontal placement may be given.

function addImageLink( tableRow, columnNum, functionName, cellNum, actionDesc,
		       imageName, backgroundColour, imageHeight, imageWidth,
		       horizontalAlign, verticalAlign )
{
    var linkNode, newImage, newCell

    // Add the cell to the specified table.
    newCell = tableRow.insertCell( columnNum )

    // Create the image element using the name given.
    newImage = document.createElement( "img" )
    newImage.setAttribute( "src", "Data/" + imageName )
    newImage.setAttribute( "border", "0" )

    // Set alternate text for image.
    if ( actionDesc == null )
    {
	newImage.setAttribute( "alt", "Cell image" )
    }
    else
    {
	newImage.setAttribute( "alt", actionDesc )
    }

    // If a function is to be called...
    if ( functionName != null )
    {
	// Create the link for it, with the specified value as a parameter.
	linkNode = document.createElement( 'a' )
	linkNode.href = "javascript:" + functionName + "(" + cellNum + ")"
	if ( actionDesc != null )
	{
	    linkNode.title = actionDesc
	}

	// Add the image to the link.
	linkNode.appendChild( newImage )

	// If a template cell that may be edited, give the cell an ID and
	//  highlight it if it is locked.
	if ( functionName == "setCell" )
	{
	    newCell.id = "CELL" + cellNum.replace( /,/, "_" )

	    if ( backgroundColour != null )
	    {
		newCell.style.background = backgroundColour
//newCell.style.borderStyle = "solid"
//newCell.style.borderColor = backgroundColour + " " + backgroundColour
	    }
	}
    }

    // Apply any specified height, width and alignment to the cell.
    if ( imageHeight > 0 )
    {
	newCell.setAttribute( "valign", "top" )
	newCell.setAttribute( "height", imageHeight )
    }
    if ( imageWidth > 0 )
    {
	newCell.setAttribute( "width", imageWidth )
    }
    if ( horizontalAlign != null )
    {
	newCell.setAttribute( "align", horizontalAlign )
    }
    if ( verticalAlign != null )
    {
	newCell.setAttribute( "valign", verticalAlign )
    }

    // Add the link or image to the table cell as appropriate.
    if ( functionName == null )
    {
	newCell.appendChild( newImage )
    }
    else
    {
	newCell.appendChild( linkNode )
    }
}



// Returns the name of the image which corresponds to the template cell at
// the specified location.

function getImageName( rowNum, columnNum )
{
    var typeKey

    // Translate the cell type into the corresponding image type.
    var cellIndex = ( rowNum * globalWidth ) + columnNum
    var cellType = globalTemplate.substring( cellIndex, cellIndex + 1 )
    switch ( cellType )
    {
	case ' ' :
	case '^' :
	    typeKey = 'E'
	    break
	case '*' :
	case ';' :
	    typeKey = 'C'
	    break
	case '#' :
	case '%' :
	    typeKey = '00'
	    break
	default :
	    showError( "Invalid cell type (" + cellType + ")", false )
	    return null
    }

    // Return the image name.
    return 'ic' + typeKey + '.bmp'
}



// Creates a new template, with the default height/width.  The template is
// actually created on the server and the result is simply loaded from there.

function createTemplate()
{
    // Ask the server to produce a new, untitled template.
    serverRequest( "New template", "POST", "Form=New" )
}



// Makes a copy of the current template.  The change is actually made on the
// server and the result is simply loaded from here.

function duplicateTemplate()
{
    serverRequest( "New template", "POST",
	       "Form=New&Title=" + globalTemplateList[ globalActiveTemplate ] )
}



// Deletes the selected template, following confirmation from the user.  The
// template is removed from the server.

function deleteTemplate( templateNum )
{
    // If the user confirms that the row is to be deleted, remove it.
    if ( confirm( "Really delete this template?" ) )
    {
	serverRequest( "Delete", "POST", "Form=Old&Title=" +
					globalTemplateList[ templateNum - 1 ] )
    }
}



// Adds a new row to the template, at the specified position.

function addRow( newRowNum )
{
    var topChar, bottomChar

    // Extract the rows that appear either side of the one to be added.
    var prevRow = globalTemplate.substring( ( newRowNum - 1 ) * globalWidth,
					    newRowNum * globalWidth )
    var nextRow = globalTemplate.substring( newRowNum * globalWidth,
					    ( newRowNum + 1 ) * globalWidth )

    // Generate a new row, linking together any floor cells (including those
    // containing a box/goal) that were already linked.  Extend the outer wall
    // and retain chaining between locked cells.
    var newRow = ""
    for ( var columnNum = 0; columnNum < globalWidth; columnNum++ )
    {
	if ( columnNum == 0 || columnNum == globalWidth - 1 )
	{
	    newRow += "%"
	}
	else
	{
	    topChar = prevRow.substring( columnNum, columnNum + 1 )
	    bottomChar = nextRow.substring( columnNum, columnNum + 1 )

	    if ( topChar == "%" && bottomChar == "%" )
	    {
		newRow += "%"
	    }
	    else if ( topChar == "#" || bottomChar == "#" )
	    {
		newRow += "#"
	    }
	    else if ( ( topChar == "^" || topChar == ";" ) &&
		      ( bottomChar == "^" || bottomChar == ";" ) )
	    {
		newRow += "^"
	    }
	    else
	    {
		newRow += " "
	    }
	}
    }

    // Separate the existing rows that appear above/below the one to be added,
    // then re-combine them with the new row in the middle.
    globalTemplate = globalTemplate.substring( 0, newRowNum * globalWidth ) +
		     newRow +
		     globalTemplate.substring( newRowNum * globalWidth )

    // Include the new row in total height and re-display the whole table.
    globalHeight++
    displayTable()

    // Add the action to the undo/redo history.
    storeAction( "+-", newRowNum, true )
}



// Deletes a row from the template.

function deleteRow( deadRowNum )
{
    // Do nothing if the table is already too small.
    if ( globalHeight <= 4 )
    {
	return
    }

    // Copy the rows that appear before and after the one being removed,
    // but exclude that row from the new version of the template.
    globalTemplate =
	globalTemplate.substring( 0, deadRowNum * globalWidth ) +
	globalTemplate.substring( ( deadRowNum + 1 ) * globalWidth )

    // Discount deleted row from total height, re-display the whole table.
    globalHeight--
    displayTable()

    // Add the action to the undo/redo history.
    storeAction( "--", deadRowNum, true )
}



// Adds a new column to the template, at the specified position.

function addColumn( newColumnNum )
{
    var rowNum, columnNum
    var oldChar, nextChar

    // For each row/column...
    var newTemplate = ""
    var index = 0
    for ( rowNum = 0; rowNum < globalHeight; rowNum++ )
    {
	for ( columnNum = 0; columnNum < globalWidth; columnNum++ )
	{
	    // Store current character, advance to the next cell.
	    oldChar = globalTemplate.substring( index, index + 1 )
	    index++

	    // Copy the existing cell to the new template.
	    newTemplate += oldChar

	    // If the insertion column has been reached, generate a new cell
	    // for it, linking together any floor cells (including those
	    // containing a box/goal) that were already linked, linking locks.
	    if ( columnNum == newColumnNum - 1 )
	    {
		nextChar = globalTemplate.substring( index, index + 1 )

		if ( oldChar == "%" && nextChar == "%" )
		{
		    newTemplate += "%"
		}
		else if ( oldChar == "#" || oldChar == "%" ||
			  nextChar == "#" || nextChar == "%" )
		{
		    newTemplate += "#"
		}
		else if ( ( oldChar == "^" || oldChar == ";" ) &&
			  ( nextChar == "^" || nextChar == ";" ) )
		{
		    newTemplate += "^"
		}
		else
		{
		    newTemplate += " "
		}
	    }
	}
    }

    // Replace the template with the new one, include new column in width.
    globalTemplate = newTemplate
    globalWidth++

    // Re-display the whole table.
    displayTable()

    // Add the action to the undo/redo history.
    storeAction( "+|", newColumnNum, true )
}



// Deletes a specified column from the template.

function deleteColumn( deadColumnNum )
{
    var rowNum, columnNum
    var oldChar

    // Do nothing if the table is already too small.
    if ( globalWidth <= 4 )
    {
	return
    }

    // For each row/column...
    var newTemplate = ""
    var index = 0
    for ( rowNum = 0; rowNum < globalHeight; rowNum++ )
    {
	for ( columnNum = 0; columnNum < globalWidth; columnNum++ )
	{
	    // Copy the cell, but only if it isn't in column being deleted.
	    if ( columnNum != deadColumnNum )
	    {
		newTemplate += globalTemplate.substring( index, index + 1 )
	    }

	    // Advance to the next cell.
	    index++
	}
    }

    // Replace template with the new one, exclude deleted column from width.
    globalTemplate = newTemplate
    globalWidth--

    // Re-display the whole table.
    displayTable()

    // Add the action to the undo/redo history.
    storeAction( "-|", deadColumnNum, true )
}



// Changes the active cell type, to be used for altering existing cells.

function setType( cellType )
{
    // If a type is already highlighted, remove the highlight or just return
    // if the type is already the one selected.
    if ( globalCellType != null )
    {
	if ( cellType == globalCellType )
	{
	    return
	}
	document.getElementById(
		"TYPE" + getTypeID( globalCellType ) ).style.backgroundColor =
				"#FFFFFF"
    }

    // Highlight the selected cell type.
    document.getElementById(
	    "TYPE" + getTypeID( cellType ) ).style.backgroundColor = "#FC0FC0"

    // Remember the cell type and its ID suffix.
    globalCellType = cellType
}



// Alters the specified cell within the template to have the active cell type.

function setCell( rowNum, columnNum )
{
    var newImage, linkNode

    // Get the existing cell type.
    var index = ( rowNum * globalWidth ) + columnNum
    var oldType = globalTemplate.substring( index, index + 1 )

    // If the action is to lock or unlock the cell, work out what the new
    // cell type will be from the existing type...
    var newType = oldType
    if ( globalCellType == 'L' )
    {
	switch ( oldType )
	{
	    case " " :
		newType = '^'
		break
	    case "*" :
		newType = ';'
		break
	    case "#" :
		newType = '%'
		break
	    default :
//		showError( "Invalid cell type (" + oldType + ")", false )
		return
	}
    }
    else if ( globalCellType == 'U' )
    {
	switch ( oldType )
	{
	    case "^" :
		newType = ' '
		break
	    case ";" :
		newType = '*'
		break
	    case "%" :
		newType = '#'
		break
	    default :
//		showError( "Invalid cell type (" + oldType + ")", false )
		return
	}
    }
    // Otherwise, if the action is to set it to a specific type, use the
    // selected type but inherit any lock from the existing cell type if not
    // replaying from history.
    else
    {
	newType = globalCellType
	if ( ( oldType == '^' || oldType == ';' || oldType == '%' ) &&
	     ! globalReplay )
	{
	    switch ( newType )
	    {
		case " " :
		    newType = '^'
		    break
		case "*" :
		    newType = ';'
		    break
		case "#" :
		    newType = '%'
		    break
		default :
		    showError( "Invalid cell type (" + newType + ")", false )
		    return
	    }
	}
    }

    // If the cell already has the selected type...
    if ( newType == oldType )
    {
	// If the last action was to change the type of this cell, undo it.
	if ( globalActionNum > 0 &&
	     globalActionList[ globalActionNum - 1 ].actionType == "=" )
	{
	    var actionValues =
		    globalActionList[ globalActionNum - 1 ].cellID.split( "&" )
	    if ( parseInt( actionValues[ 1 ], 10 ) == rowNum &&
		 parseInt( actionValues[ 2 ], 10 ) == columnNum )
	    {
		replayActions( true )
	    }
	}
	return
    }

    // Copy the cells before and after the one being updated, replacing the
    // single cell with its new value in between.
    globalTemplate = globalTemplate.substring( 0, index ) + newType +
		     globalTemplate.substring( index + 1 )

    // If a new action, not the result of an undo/redo...
    if ( ! globalReplay )
    {
	// Create the new image for the cell.
	newImage = document.createElement( "img" )
	newImage.setAttribute( "src", "Data/" +
				      getImageName( rowNum, columnNum ) )
	newImage.setAttribute( "border", "0" )

	// Create the link for it, with the specified value as a parameter.
	linkNode = document.createElement( 'a' )
	linkNode.href = "javascript:setCell(" + rowNum + "," + columnNum + ")"

	// Add the image to the link.
	linkNode.appendChild( newImage )

	// Replace the old image with the new one.
	oldCell = document.getElementById( "CELL" + rowNum + "_" + columnNum )
	oldCell.removeChild( oldCell.firstChild )
	oldCell.appendChild( linkNode )

	// Indicate if the cell is locked or not.
	if ( newType == '^' || newType == ';' ||  newType == '%' )
	{
	    oldCell.style.background = "#FF0000"
//oldCell.style.borderStyle = "solid"
//oldCell.style.borderColor = "#FF0000"
	}
	else
	{
	    oldCell.style.background = "#FFFFFF"
	}

	// Add the action to the undo/redo history.
	storeAction( "=", newType + "&" + rowNum + "&" + columnNum, true )
    }
}



// Adds a new action to the undo/redo history.

function storeAction( actionType, cellID, saveChange )
{
    // If a new action, not undoing an old one...
    if ( ! globalReplay )
    {
	// Add the action at current position and advance.
	globalActionList[ globalActionNum++ ] =
			new actionInfo( actionType, cellID )

	// Add action to total, discard any redo actions.
	globalNumActions = globalActionNum

	// Make the undo link/image visible, hide the redo link/image.
	setReplayState()

	// Trigger server update, unless just loading a template.
	if ( saveChange )
	{
	    saveTemplate()
	}
    }
}



// Takes back the last action in the undo/redo history, making any current
// action available to be re-done.

function replayActions( isUndo )
{
    // If an undo was requested and there are actions prior to the current
    // state, or a redo was requested and there are actions after...
    if ( ( isUndo && globalActionNum > 0 ) ||
	 ( ! isUndo && globalActionNum < globalNumActions ) )
    {
	// Block display changes and additions to the history during the undo.
	globalReplay = true

	// Restore the template to its initial state.
	globalTemplate = globalInitialTemplate
	globalHeight = globalInitialHeight
	globalWidth = globalInitialWidth

	// Remember the active cell type.
	var cellType = globalCellType

	// Step back or forwards in the history as appropriate.
	if ( isUndo )
	{
	    globalActionNum--
	}
	else
	{
	    globalActionNum++
	}

	// For each action required to get to the new state...
	for ( actionNum = 0; actionNum < globalActionNum; actionNum++ )
	{
	    // Determine which cells were affected by the action.
	    var cellID = globalActionList[ actionNum ].cellID

	    // Perform the action again, based on its type.
	    var actionType = globalActionList[ actionNum ].actionType
	    switch ( actionType )
	    {
		case "+-" :
		    addRow( parseInt( cellID, 10 ) )
		    break
		case "--" :
		    deleteRow( parseInt( cellID, 10 ) )
		    break
		case "+|" :
		    addColumn( parseInt( cellID, 10 ) )
		    break
		case "-|" :
		    deleteColumn( parseInt( cellID, 10 ) )
		    break
		case "=" :
		    var actionValues = cellID.split( "&" )
		    globalCellType = actionValues[ 0 ]
		    setCell( parseInt( actionValues[ 1 ], 10 ),
			     parseInt( actionValues[ 2 ], 10 ) )
		    break
		default :
		    showError( "Invalid action (" + actionType + ")", false )
		    return
	    }
	}

	// Restore the active cell type.
	globalCellType = cellType

	// Remove block on the display and additions to the history.
	globalReplay = false

	// Re-display the whole table.
	displayTable()

	// Update the visibility of the undo/redo images/links.
	setReplayState()

	// Trigger server update.
	saveTemplate()
    }
}



// Adds a template to the list of available templates at the specified
// position, either for display or (for the active template) as an edit box
// to allow it to be renamed.

function addTemplate( templateNum, editMode )
{
    var textNode

    // Add link to allow the template to be deleted.
    var newRow = globalList.insertRow( templateNum + 2 )
    addImageLink( newRow, 0, "deleteTemplate", templateNum + 1,
		  "Delete template", "delete.gif", null, 0, 0, null, null )

    // Add the template name to the list, with a link to select it.
    var newCell = newRow.insertCell( 1 )
    newCell.setAttribute( "width", "100%" )
    newCell.setAttribute( "align", "left" )
    if ( editMode )
    {
	var newElement = document.createElement( "input" )
	newElement.id = "NameEditor"
	newElement.type = "text"
	newElement.size = globalTitleSize
	newElement.maxLength = globalTitleSize
	newElement.value =
		decodeURIComponent( globalTemplateList[ templateNum ] )
	newElement.style.fontFamily = "monospace"
	newElement.style.fontSize = "x-large"
	newElement.onkeypress = keyPressTest
	newElement.onblur = performRename
	newCell.appendChild( newElement )
	newElement.focus()
    }
    else
    {
	var linkNode = document.createElement( 'a' )
	linkNode.style.textDecoration = "none"
	if ( templateNum == globalActiveTemplate )
	{
	    linkNode.href = "javascript:renameTemplate()"
	    newCell.onclick = function () { renameTemplate() }
	    newCell.style.background = "#FFD1DC"
	}
	else
	{
	    linkNode.style.color = "silver"
	    linkNode.href = "javascript:goToTemplate(" + templateNum + ")"
	}
	textNode = document.createTextNode(
		      decodeURIComponent( globalTemplateList[ templateNum ] ) )
	linkNode.appendChild( textNode )
	newCell.style.fontFamily = "monospace"
	newCell.style.fontSize = "x-large"
	newCell.appendChild( linkNode )
    }

    // Create hidden copy to align the main editor in the centre of the screen.
    newRow = globalListCopy.insertRow( templateNum + 1 )
    newRow.insertCell( 0 )
    newCell = newRow.insertCell( 1 )
    newCell.setAttribute( "width", "100%" )
    textNode = document.createTextNode(
		      decodeURIComponent( globalTemplateList[ templateNum ] ) )
    newCell.style.fontFamily = "monospace"
    newCell.style.fontSize = "x-large"
    newCell.appendChild( textNode )
}



// Converts the active template name into an edit box, allowing the user to
// rename it.

function renameTemplate()
{
    // Remove the existing entry for the active template from list of names.
    globalList.deleteRow( globalActiveTemplate + 2 )
    globalListCopy.deleteRow( globalActiveTemplate )

    // Convert the active template name into an edit box.
    addTemplate( globalActiveTemplate, true )
}



// Used to finish editing the current template name when the escape key or
// return key is pressed within it.

function keyPressTest( eventNode, objectNode )
{
    // Finish editing if the escape key or return key was pressed.
    keyCode = (window.event) ? event.keyCode : eventNode.keyCode
    escapeKey = (window.event) ? 27 : eventNode.DOM_VK_ESCAPE
    if ( keyCode == escapeKey || keyCode == 13 )
    {
	performRename()
    }
}



// Performs the actual renaming of a template after the new name is entered.

function performRename()
{
    // If the rename has not already finished...
    var nameEditor = document.getElementById( "NameEditor" )
    if ( nameEditor != null )
    {
	// If the name has changed...
	var oldName =
		globalTemplateList[ globalActiveTemplate ].replace( /\n/g, "" )
	var newName =
		encodeURIComponent( nameEditor.value.replace( / +$/, "" ) )
	if ( newName != oldName )
	{
	    // Don't permit the name to be changed if it is already in use.
	    for ( var index = 0; index < globalTemplateList.length; index++ )
	    {
		if ( globalTemplateList[ index ] == newName )
		{
		    return
		}
	    }

	    // Rename the file on the server.
	    serverRequest( "Rename", "POST",
			   "Form=Rename&Title=" + oldName + "&New=" + newName )

	    // Store the new name.
	    globalTemplateList[ globalActiveTemplate ] = newName
	}

	// Remove the existing entry for the active template from list of names.
	globalList.deleteRow( globalActiveTemplate + 2 )
	globalListCopy.deleteRow( globalActiveTemplate )

	// Re-display active template name as text, with link to rename again.
	addTemplate( globalActiveTemplate, false )
    }
}



// Updates the server with the current template definition and its associated
// undo/redo history.

function saveTemplate()
{
    var actionType, cellID

    // Convert the actions to the undo/redo history into a request string,
    // starting with the height/width of the initial, empty grid.
    var actionList = ( globalInitialHeight - 2 ) + "&" +
		     ( globalInitialWidth - 2 )
    for ( var index = 0; index < globalNumActions; index++ )
    {
	actionType = globalActionList[ index ].actionType
	cellID = globalActionList[ index ].cellID

	if ( actionType == "=" )
	{
	    actionList += " [" + encodeURIComponent( cellID.substring( 0, 1 ) )
			 + "]&" + cellID.substring( 2 )
	}
	else
	{
	    actionList += " " + actionType.replace( /\+/g, "%2B" ) + cellID
	}
    }

    // Send the template definition and undo/redo history to the server.
    serverRequest( "Save", "POST",
			 "Form=Save&Title=" +
				globalTemplateList[ globalActiveTemplate ] +
			 "&Height=" + ( globalHeight - 2 ) +
			 "&Width=" + ( globalWidth - 2 ) +
			 "&Definition=" + encodeURIComponent( globalTemplate ) +
			 "&ActionNum=" + ( globalActionNum + 1 ) +
			 "&NumActions=" + ( globalNumActions + 1 ) +
			 "&History=" + encodeURIComponent( actionList ) )
}



// Makes the undo/redo images/links visible or hidden as appropriate.

function setReplayState()
{
    // If actions precede the current one, make undo link/image visible...
    if ( globalActionNum > 0 )
    {
	globalUndo.style.visibility = "visible"
    }
    // Otherwise, make sure that it is hidden.
    else
    {
	globalUndo.style.visibility = "hidden"
    }

    // If actions follow the current one, make the redo link/image visible...
    if ( globalActionNum < globalNumActions )
    {
	globalRedo.style.visibility = "visible"
    }
    // Otherwise, make sure that it is hidden.
    else
    {
	globalRedo.style.visibility = "hidden"
    }
}



// Increases or decreases the spacing between cells.

function setSpacing( makeLarger )
{
    // If the user requested a greater spacing...
    if ( makeLarger )
    {
	// Do nothing if the spacing is already very large.
	if ( globalCellSize + 2 > globalConfigCellSize * 2 )
	{
	    return
	}

	// Increase the spacing.
	globalCellSize += 2
    }
    // Otherwise, if the user requested a smaller spacing...
    else
    {
	// Do nothing if there is already no spacing present.
	if ( globalCellSize - 2 < globalConfigCellSize )
	{
	    return
	}

	// Decrease the spacing.
	globalCellSize -= 2
    }

    // Store the new setting in a browser cookie.
    setCookie( "CELLSIZE", globalCellSize )

    // Re-display the table with the new spacing.
    globalDisplayHeight = 0
    globalDisplayWidth = 0
    displayTable()
}



// Returns the ID letter which corresponds to the specified Sokoban cell type.

function getTypeID( cellType )
{
    // Base return value on cell type.
    switch ( cellType )
    {
	case ' ' :
	    return 'A'
	case '*' :
	    return 'F'
	case '#' :
	    return 'G'
	case 'L' :
	case 'U' :
	    return cellType
    }

    // Fail if type is invalid.
    showError( "Invalid cell type (" + cellType + ")", false )
    return null
}



// Displays the specified text as the current status, using the given colour
// values.  If the message regards the completion of a task, the fade option
// should be used to remove the message after a brief delay.

function showStatus( message, redValue, greenValue, blueValue, wantFade )
{
    var textNode

    // Discard any existing status text and associated timer.
    while ( globalStatusNode.firstChild != null )
    {
	globalStatusNode.removeChild( globalStatusNode.firstChild )
    }
    if ( globalStatusTimer != null )
    {
	clearTimeout( globalStatusTimer )
    }

    // Set style of status node.
    globalStatusNode.style.fontFamily = "monospace"
    globalStatusNode.style.fontWeight = "bold"
    globalStatusNode.style.color = "white"

    // Set background colour using the values specified.
    globalStatusRed = redValue
    globalStatusGreen = greenValue
    globalStatusBlue = blueValue
    globalStatusNode.style.backgroundColor = "rgb(" + globalStatusRed + "," +
						      globalStatusGreen + "," +
						      globalStatusBlue + ")"

    // Put the text provided into the status node.
    textNode = document.createTextNode( '\u00a0' + message + '\u00a0' )
    globalStatusNode.appendChild( textNode )

    // Start fading the status to white, if required.
    if ( wantFade )
    {
	globalStatusTimer = setTimeout( "fadeStatus()", 100 )
    }
}



// Displays the given text as an error, and blocks further edits.

function showError( message, fromServer )
{
    // Indicate that an error has occurred.
    if ( fromServer )
    {
	showStatus( "Server error", 255, 0, 0, 0 )
    }
    else
    {
	showStatus( "Error", 255, 0, 0, 0 )
    }

    // Hide all of the editing tables/elements.
    globalList.style.visibility = "hidden"
    globalRowAdder.style.visibility = "hidden"
    globalColumnAdder.style.visibility = "hidden"
    globalColumnRemover.style.visibility = "hidden"
    globalDataTable.style.visibility = "hidden"
    document.getElementById( 'Controller' ).style.visibility = "hidden"

    // Inform the user of the next action to take.
    document.getElementById( "Title" ).innerHTML =
	'<P><TT><B>Please reload this page to continue.</B></TT></P>'

    // Show the error message, setting its foreground colour.
    while ( globalWarningNode.firstChild != null )
    {
	globalWarningNode.removeChild( globalWarningNode.firstChild )
    }
    textNode = document.createTextNode( message )
    globalWarningNode.appendChild( textNode )

    // Make error message visible.
    globalWarningNode.style.color = "black"
}



// Repeatedly changes the background colour of the status text, gradually
// changing it to match the background so that it becomes invisible.

function fadeStatus( fromTimer )
{
    // Increase the colour up to the maximum.
    globalStatusRed += 10
    globalStatusGreen += 10
    globalStatusBlue += 10
    if ( globalStatusRed > 255 )
    {
	globalStatusRed = 255
    }
    if ( globalStatusGreen > 255 )
    {
	globalStatusGreen = 255
    }
    if ( globalStatusBlue > 255 )
    {
	globalStatusBlue = 255
    }

    // Change the background colour of the status text.
    globalStatusNode.style.backgroundColor = "rgb(" + globalStatusRed + "," +
						      globalStatusGreen + "," +
						      globalStatusBlue + ")"

    // If maximum not reached, do another colour change after a short delay...
    if ( globalStatusRed < 255 ||
	 globalStatusGreen < 255 || globalStatusBlue < 255 )
    {
	globalStatusTimer = setTimeout( "fadeStatus()", 100 )
    }
    // Otherwise, remove the text.
    else
    {
	globalStatusNode.removeChild( globalStatusNode.firstChild )
    }
}



// Adds or updates a cookie name/value pair, which will expire after 30 days.

function setCookie( name, value )
{
    var expiryDate

    // Calculate the expiry date.
    expiryDate = new Date()
    expiryDate.setTime( expiryDate.getTime() + ( 30 * 86400000 ) )

    // Set cookie, with name/value pair and expiry date.
    document.cookie = name + "=" + value +
		      "; expires=" + expiryDate.toGMTString() + "; path=/"
}



// Returns the value of a specified name from cookie, if defined.

function readCookie( name )
{
    var valueList, item, index

    // Get the cookie, convert it into a list.
    valueList = document.cookie.split( ';' )

    // For each item in the list...
    for ( index = 0; index < valueList.length; index++ )
    {
	item = valueList[ index ]

	// Discard leading spaces.
	while ( item.charAt( 0 ) == ' ' )
	{
	    item = item.substring( 1, item.length )
	}

	// If the item matches, return the value part.
	if ( item.indexOf( name + "=" ) == 0 )
	{
	    return item.substring( name.length + 1, item.length )
	}
    }

    // Return nothing if no matches were found.
    return null
}



// Returns the current date/time according to the browser, in the format
// YYYYMMDDHHMMSS, followed by a random number (typically around 14-17 digits).
// The result is used to trivially avoid all caching of dynamic server data.

function getUniqueID()
{
    // Get the local time from the browser.
    var currentTime = new Date()

    // Get the day number and month number.
    var dayNum = currentTime.getDay() + 1
    var monthNum = currentTime.getMonth() + 1

    // Split the time into hours, minutes and seconds.
    var numHours = currentTime.getHours()
    var numMinutes = currentTime.getMinutes()
    var numSeconds = currentTime.getSeconds()

    // Format as a date/time and return result.
    return "" + currentTime.getFullYear() +
		( ( monthNum < 10 ) ? "0" : "" ) + monthNum +
		( ( dayNum < 10 ) ? "0" : "" ) + dayNum +
		( ( numHours < 10 ) ? "0" : "" ) + numHours +
		( ( numMinutes < 10 ) ? "0" : "" ) + numMinutes +
		( ( numSeconds < 10 ) ? "0" : "" ) + numSeconds +
		( Math.random() + "" ).substring( 2 )
}



// Object used to hold an individual entry in the server request queue.

function queueInfo( requestType, requestMethod, requestData )
{
    this.requestType = requestType
    this.requestMethod = requestMethod
    this.requestData = requestData
}



// Object used to hold an individual entry in the undo/redo history.

function actionInfo( actionType, cellID )
{
    this.actionType = actionType
    this.cellID = cellID
}

