//*****************************************************************************
//
//	Lee Haywood's SokME - a Sokoban/Multiban/Hexoban game.
//	Copyright (C) 2002, Lee J Haywood.
//
//	Sokoban is a copyright of Thinking Rabbit Inc., Japan.
//
//	This program is free software; you can redistribute it and/or
//	modify it under the terms of the GNU General Public License
//	as published by the Free Software Foundation; either version 2
//	of the License, or (at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to:
//
//		The Free Software Foundation, Inc.,
//		59 Temple Place - Suite 330, Boston, MA
//		02111-1307, USA.
//
//*****************************************************************************

import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.net.*;
import java.lang.*;
import java.util.*;
import java.io.*;



// Class that implements the SokME game program.

public class sokme extends Applet implements AdjustmentListener
{
    final int HexagonInfo[] =
	{
	    20,21, 18,22, 16,24, 15,26, 12,28, 11,30, 8,32, 6,34, 5,36,
	    2,37, 1,39, 0,40, 0,40, 0,40, 0,40, 0,40, 0,40, 0,40, 0,40,
	    0,40, 0,40, 0,40, 0,40, 0,40, 0,40, 0,40, 0,40, 0,40, 0,40,
	    0,40, 0,40, 0,40, 0,40, 0,40, 0,40, 1,38, 3,36, 5,35, 7,32,
	    9,31, 11,28, 13,26, 15,25, 17,22, 18,21
	};
    static String ErrorMessage;
    static int DisplayHeight, DisplayWidth;
    static int CellHeight, CellWidth, CellVerticalOffset;
    static int MaskHeight, MaskWidth;
    static int Rows, Columns;
    static int GameTop, GameLeft, MenuTop;
    private boolean PlayingGame;
    private boolean IsHexoban;
    private boolean IsFirstGame;
    private Graphics Display;
    private BorderLayout Sections;
    private Panel Options;
    private FontMetrics FontInfo;
    private int FontHeight, FontAscent;
    private int NumDesigns;
    private int NumUnmatched;
    private Scrollbar ScrollBar;
    private Vector TitleList;
    private Vector DesignList;
    private boolean HaveCompleted[];
    private Vector UndoList, RedoList;
    private Image MarbleImage, WoodImage, TargetImage, MatchImage;
    private Image PusherImage, BlobImage, MultibanImage;
    private int CurrentDesign, MenuHeight, MenuFirstDesign;
    private int PlayerRow, PlayerColumn;
    private char[][] Arena;
    private int[][] SearchArea;
    private int SearchDistance;
    private Button Restart, Undo, Redo, Quit;



    // ***** EXTERNALLY-CALLED (PUBLIC) METHODS.



    // Initialises the program and displays design selection menu.

    public void init()
    {
	String Filename, NewTitle;
	URL DataURL;
	BufferedReader Handle;
	Vector NewDesign;
	String InputLine;
	int Row, Index;
	boolean ExpectDesign, IsDefinition;

	// Display no error message by default.
	ErrorMessage = null;

	// Determine the resolution of the display window.
	DisplayHeight = this.bounds().height;
	DisplayWidth = this.bounds().width;

	// Create a separate region for cells to be drawn in.
	Sections = new BorderLayout();
	setLayout( Sections );

	// Create a panel of buttons.
	Restart = new Button( "Restart" );
	Undo = new Button( "Undo" );
	Redo = new Button( "Redo" );
	Options = new Panel();
	Options.add( Restart );
	Options.add( Undo );
	Options.add( Redo );

	// Determine where the graphics should be drawn.
	Display = getGraphics();

	// Obtain information for the current font.
	FontInfo = Display.getFontMetrics();
	FontHeight = FontInfo.getHeight();
	FontAscent = FontInfo.getAscent();

	// Calculate how many menu entries can be displayed at a time.
	MenuHeight = DisplayHeight / FontHeight;

	// Get the designs and store their titles...
	try
	{
	    // Create title and design lists dynamically.
	    TitleList = new Vector();
	    DesignList = new Vector();

	    // Open the design definition file using the network.
	    Filename = this.getParameter( "designs" );
	    if ( Filename == null )
	    {
		Filename = "sokevo.txt";
	    }
	    DataURL = new URL( getCodeBase(), Filename );
	    Handle = new BufferedReader(
			    new InputStreamReader( DataURL.openStream() ) );

	    // Assume that the file contains a Hexoban collection if it has a
	    // final extension of 'hsb' (using any letter case).
	    IsHexoban = Filename.toLowerCase().endsWith( ".hsb" );

	    // For each line in the file...
	    NumDesigns = 0;
	    ExpectDesign = false;
	    NewTitle = null;
	    NewDesign = null;
	    while ( ( InputLine = Handle.readLine() ) != null )
	    {
		// If the line is blank then the next non-blank line will be the
		// start of a new design definition...
		if ( InputLine.length() <= 0 )
		{
		    ExpectDesign = true;
		    NewTitle = null;
		    NewDesign = null;
		}
		// Otherwise, if a comment or definition line...
		else if ( ExpectDesign )
		{
		    // Determine if the line represents part of a definition or
		    // not.  If not, it is assumed to be a comment line.
		    IsDefinition = true;
		    for ( Index = 0; Index < InputLine.length(); Index++ )
		    {
			switch ( InputLine.charAt( Index ) )
			{
			    case ' ' :
			    case '#' :
			    case '@' :
			    case '+' :
			    case '.' :
			    case '$' :
			    case '*' :
				break;
			    default :
				IsDefinition = false;
			}
		    }

		    // If the line is part of a definition...
		    if ( IsDefinition )
		    {
			// For the first line, generate an entry to hold the
			// new design, storing its title if provided.
			if ( NewDesign == null )
			{
			    if ( NewTitle == null )
			    {
				NewTitle = "Untitled";
			    }
			    TitleList.addElement( NewTitle );
			    NewDesign = new Vector();
			    DesignList.addElement( NewDesign );
			    NumDesigns++;
			}

			// Add the line to the current definition entry.
			NewDesign.addElement( InputLine );
		    }
		    // Otherwise, if the line is a comment...
		    else
		    {
			// If comment follows a design, stop adding to it...
			if ( NewDesign != null )
			{
			    NewDesign = null;
			    ExpectDesign = false;
			}
			// Otherwise, store first comment line as design title.
			// Strip any leading ';' and surrounding whitespace.
			else if ( NewTitle == null )
			{
			    if ( InputLine.length() > 1 &&
				 InputLine.charAt( 0 ) == ';' )
			    {
				NewTitle = InputLine.substring( 2 ).trim();
			    }
			    else
			    {
				NewTitle = InputLine.trim() ;
			    }
			}
		    }
		}
	    }

	    // Close the design definition file.
	    Handle.close();
	}

	// Handle inability to connect to server.
	catch ( MalformedURLException Exception )
	{
	    ErrorMessage = "Failed to access design definition URL.";
	    return;
	}

	// Handle failure to read from server.
	catch ( IOException Exception )
	{
	    ErrorMessage = "Failed to read design definition file.";
	    return;
	}

	// If no titles were found...
	if ( NumDesigns <= 0 )
	{
	    // Read the whole file as one design.
	    try
	    {
		DataURL = new URL( getCodeBase(), Filename );
		Handle = new BufferedReader(
			    new InputStreamReader( DataURL.openStream() ) );

		NewDesign = new Vector();
		while ( ( InputLine = Handle.readLine() ) != null )
		{
		    NewDesign.addElement( InputLine );
		}
		Handle.close();
		TitleList.addElement( Filename );
		DesignList.addElement( NewDesign );
		NumDesigns = 1;
	    }

	    // Handle inability to connect to server.
	    catch ( MalformedURLException Exception )
	    {
		ErrorMessage = "Failed to access design definition URL.";
		return;
	    }

	    // Handle failure to read from server.
	    catch ( IOException Exception )
	    {
		ErrorMessage = "Failed to read design definition file.";
		return;
	    }
	}

	// Fail if no designs were loaded.
	if ( NumDesigns <= 0 )
	{
	    ErrorMessage = "No designs found in definition file.";
	    return;
	}

	// Create a completion indicator for each design.
	HaveCompleted = new boolean[ NumDesigns ];

	// If there are few designs, centre the menu vertically.
	if ( NumDesigns < MenuHeight )
	{
	    MenuTop = ( DisplayHeight - ( NumDesigns * FontHeight ) ) / 2;
	}
	// Otherwise, display the menu from the top with a scroll bar.
	else
	{
	    MenuTop = 0;
	    ScrollBar = new Scrollbar( Scrollbar.VERTICAL, 0,
				       MenuHeight, 0, NumDesigns );
	    ScrollBar.addAdjustmentListener( this );
	    add( "East", ScrollBar );
	}

	// Start with the first design.
	CurrentDesign = 0;

	// Require images to be loaded before their first use.
	IsFirstGame = true;
	MultibanImage = null;

	// If there is only one design, play it immediately...
	if ( NumDesigns == 1 )
	{
	    PlayingGame = true;
	    New_Game( true, false );
	}
	// Otherwise, start in menu mode and draw the selection menu.  Add a
	// button to the options panel allowing return to the menu.
	else
	{
	    Quit = new Button( "Quit" );
	    Options.add( Quit );
	    PlayingGame = false;
	    Draw_Menu( true );
	}
    }

    // Draws the current display when required.

    public void paint( Graphics Display )
    {
	// Display an error message if a fatal error has occurred...
	if ( ErrorMessage != null )
	{
	    Display.drawString( ErrorMessage, 0, DisplayHeight / 2 );
	}
	// If playing a design, draw the current state...
	else if ( PlayingGame )
	{
	    Draw_Arena();
	}
	// Otherwise, draw the menu and highlight active design.
	else
	{
	    Draw_Menu( false );
	}
    }

    // Moves up and down the menu when the scroll bar is used.

    public void adjustmentValueChanged( java.awt.event.AdjustmentEvent Event )
    {
	// If the scroll bar indicates a different position to that shown...
	if ( Event.getValue() != MenuFirstDesign )
	{
	    // Use the scroll bar position to set first visible design.
	    MenuFirstDesign = Event.getValue();

	    // Change the current design if the selection has scrolled off.
	    if ( CurrentDesign < MenuFirstDesign )
	    {
		CurrentDesign = MenuFirstDesign;
	    }
	    if ( CurrentDesign >= ( MenuFirstDesign + MenuHeight ) )
	    {
		CurrentDesign = ( MenuFirstDesign + MenuHeight ) - 1;
	    }

	    // Redraw the entire menu using new starting point.
	    Draw_Menu( false );
	}
    }

    // Responds to a button selection.

    public boolean action( Event Template, Object arg )
    {
	// If playing a game...
	if ( PlayingGame )
	{
	    // Undo the previous move if requested.
	    if ( Template.target == Undo )
	    {
		Undo_Move();
		return true;
	    }

	    // Redo the previous move if requested.
	    if ( Template.target == Redo )
	    {
		Redo_Move();
		return true;
	    }

	    // Re-initialise the design if restarting.
	    if ( Template.target == Restart )
	    {
		New_Game( false, true );
		return true;
	    }

	    // Go back to the menu if quitting.
	    if ( Template.target == Quit )
	    {
		remove( Options );
		invalidate();
		if ( MenuTop == 0 )
		{
		    add( "East", ScrollBar );
		    validate();
		}
		Draw_Menu( true );
		return true;
	    }
	}

	// Indicate that the event wasn't handled.
	return false;
    }

    // Handles a mouse click when an event is received.

    public boolean mouseDown( Event MouseClick, int MouseX, int MouseY )
    {
	int Row, Column;
	int TestRow, TestColumn;
	int CellTop, CellLeft;
	int OldRow, OldColumn, Index;
	boolean CellFound;

	// If playing the game...
	if ( PlayingGame )
	{
	    // If playing Hexoban, test each cell in turn for a match...
	    if ( IsHexoban )
	    {
		Column = -1;
		CellFound = false;
		for ( Row = 0; Row < Rows; Row++ )
		{
		    Column = Row % 2;
		    while ( Column < Columns )
		    {
			if ( Inside_Hexoban_Cell( Row, Column,
						  MouseY, MouseX ) )
			{
			    CellFound = true;
			    break;
			}
			Column += 2;
		    }
		    if ( CellFound )
		    {
			break;
		    }
		}
	    }
	    // Otherwise, determine which cell was selected directly.
	    else
	    {
		Row = ( MouseY - GameTop ) / CellHeight;
		Column = ( MouseX - GameLeft ) / CellWidth;
	    }

	    // If the mouse was used over the game area...
	    if ( Row >= 0 && Row < Rows && Column >= 0 && Column <= Columns )
	    {
		// If the right mouse button was used, attempt to undo...
		if ( MouseClick.modifiers == MouseClick.META_MASK )
		{
		    Undo_Move();
		}
		// Otherwise, if middle mouse button used, attempt to redo...
		else if ( MouseClick.modifiers == MouseClick.ALT_MASK )
		{
		    Redo_Move();
		}
		// Otherwise...
		else
		{
		    // If the cell is valid and contains a Multiban player,
		    // swap the player with the current player.
		    if ( Arena[ Row ][ Column ] == '@' ||
			 Arena[ Row ][ Column ] == '+' )
		    {
			Multiban_Move( Row, Column, true );
		    }
		    // Otherwise, if the cell is empty or is a target, attempt
		    // to move the player directly to it.
		    else if ( Arena[ Row ][ Column ] == ' ' ||
			      Arena[ Row ][ Column ] == '.' )
		    {
			Initialise_Search( PlayerRow, PlayerColumn, false );
			if ( Hop_Path( Row, Column ) )
			{
			    OldRow = PlayerRow;
			    OldColumn = PlayerColumn;
			    PlayerRow = Row;
			    PlayerColumn = Column;
			    Draw_Cell( OldRow, OldColumn );
			    Draw_Cell( PlayerRow, PlayerColumn );
			    UndoList.addElement( OldRow + " " + OldColumn +
						 " " + Row + " " + Column );
			    RedoList = new Vector();
			}
		    }
		    // Otherwise, if the cell contains a box...
		    else if ( Arena[ Row ][ Column ] == '$' ||
			      Arena[ Row ][ Column ] == '*' )
		    {
			// Determine the offset from the player to the box.
			Row -= PlayerRow;
			Column -= PlayerColumn;
    
			// If the box is next to the player, try to push it.
			if ( IsHexoban &&
			     ( ( ( Row == -1 || Row == 1 ) &&
			         ( Column == -1 || Column == 1 ) ) ||
			         ( Row == 0 &&
				   ( Column == -2 || Column == 2 ) ) ) ||
			     ( ( ! IsHexoban ) &&
			       ( ( ( Row == -1 || Row == 1 ) && Column == 0 ) ||
			         ( Row == 0 &&
				   ( Column == -1 || Column == 1 ) ) ) ) )
			{
			    Make_Move( Row, Column, true );
			}
		    }
		}

		// Return indicating that the event was handled.
		return true;
	    }
	}
	// Otherwise, if displaying the menu...
	else
	{
	    // If clicking over a design title...
	    Index = ( MouseY - MenuTop ) / FontHeight;
	    if ( Index >= 0 && Index < MenuHeight )
	    {
		// Determine which design was selected.
		Index += MenuFirstDesign;

		// If selecting the current design, start playing it...
		if ( Index == CurrentDesign )
		{
		    if ( MenuTop == 0 )
		    {
			remove( ScrollBar );
			invalidate();
		    }
		    New_Game( true, false );
		}
		// Otherwise, make the new selection the current design.
		else if ( Index < NumDesigns )
		{
		    Draw_Menu_Item( CurrentDesign, false );
		    CurrentDesign = Index;
		    Draw_Menu_Item( CurrentDesign, true );
		}

		// Return indicating that the event was handled.
		return true;
	    }
	}

	// Return unsuccessfully, indicating that mouse event was not handled.
	return false;
    }

    // Handles a keypress when an event is received.

    public boolean keyDown( Event Keys, int KeyCode )
    {
	// If playing a design...
	if ( PlayingGame )
	{
	    // If playing Hexoban, allow the numeric keypad to control movement.
	    if ( IsHexoban )
	    {
		switch ( KeyCode )
		{
		    case '1' :
			Make_Move( 1, -1, true );
			return true;
		    case '3' :
			Make_Move( 1, 1, true );
			return true;
		    case '4' :
			Make_Move( 0, -2, true );
			return true;
		    case '6' :
			Make_Move( 0, 2, true );
			return true;
		    case '7' :
			Make_Move( -1, -1, true );
			return true;
		    case '9' :
			Make_Move( -1, 1, true );
			return true;
		}
	    }

	    // Base action on key type.
	    switch ( KeyCode )
	    {
		// Attempt to move player when arrow keys are used.
		case Keys.UP :
		    if ( ! IsHexoban )
		    {
			Make_Move( -1, 0, true );
		    }
		    return true;
		case Keys.DOWN :
		    if ( ! IsHexoban )
		    {
			Make_Move( 1, 0, true );
		    }
		    return true;
		case Keys.LEFT :
		    Make_Move( 0, -1, true );
		    return true;
		case Keys.RIGHT :
		    Make_Move( 0, 1, true );
		    return true;
		// Go back one step if undo requested.
		case Keys.BACK_SPACE :
		case 'u' :
		    Undo_Move();
		    return true;
		// Go forward one step if redo requested.
		case Keys.INSERT :
		case 'r' :
		    Redo_Move();
		    return true;
		// Re-initialise the design if restarting.
		case 'R' :
		    New_Game( false, true );
		    return true;
		// Return to the menu if quitting.
		case 'Q' :
		    if ( NumDesigns > 1 )
		    {
			remove( Options );
			invalidate();
			if ( MenuTop == 0 )
			{
			    add( "East", ScrollBar );
			    validate();
			}
			Draw_Menu( true );
			return true;
		    }
	    }
	}
	// Otherwise, if displaying the menu...
	else
	{
	    // Allow the user to choose the previous or next design listed.
	    if ( KeyCode == Keys.UP )
	    {
		if ( CurrentDesign > 0 )
		{
		    Draw_Menu_Item( CurrentDesign, false );
		    CurrentDesign--;
		    if ( CurrentDesign < MenuFirstDesign )
		    {
			Draw_Menu( true );
		    }
		    else
		    {
			Draw_Menu_Item( CurrentDesign, true );
		    }
		}
		return true;
	    }
	    if ( KeyCode == Keys.DOWN )
	    {
		if ( CurrentDesign < NumDesigns - 1 )
		{
		    Draw_Menu_Item( CurrentDesign, false );
		    CurrentDesign++;
		    if ( CurrentDesign >= MenuFirstDesign + MenuHeight )
		    {
			Draw_Menu( true );
		    }
		    else
		    {
			Draw_Menu_Item( CurrentDesign, true );
		    }
		}
		return true;
	    }

	    // Allow the user to page up and down quickly.
	    if ( KeyCode == Keys.PGUP )
	    {
		if ( CurrentDesign > 0 )
		{
		    Draw_Menu_Item( CurrentDesign, false );
		    CurrentDesign -= MenuHeight;
		    if ( CurrentDesign < 0 )
		    {
			CurrentDesign = 0;
		    }
		    if ( CurrentDesign < MenuFirstDesign )
		    {
			Draw_Menu( true );
		    }
		    else
		    {
			Draw_Menu_Item( CurrentDesign, true );
		    }
		}
		return true;
	    }
	    if ( KeyCode == Keys.PGDN )
	    {
		if ( CurrentDesign < NumDesigns - 1 )
		{
		    Draw_Menu_Item( CurrentDesign, false );
		    CurrentDesign += MenuHeight;
		    if ( CurrentDesign >= NumDesigns )
		    {
			CurrentDesign = NumDesigns - 1;
		    }
		    if ( CurrentDesign >= MenuFirstDesign + MenuHeight )
		    {
			Draw_Menu( true );
		    }
		    else
		    {
			Draw_Menu_Item( CurrentDesign, true );
		    }
		}
		return true;
	    }

	    // Allow the user to jump to the start and end of the menu.
	    if ( KeyCode == Keys.HOME )
	    {
		if ( CurrentDesign > 0 )
		{
		    CurrentDesign = 0;
		    Draw_Menu( true );
		}
		return true;
	    }
	    if ( KeyCode == Keys.END )
	    {
		if ( CurrentDesign < NumDesigns - 1 )
		{
		    CurrentDesign = NumDesigns - 1;
		    Draw_Menu( true );
		}
		return true;
	    }

	    // Start playing when the user selects a design.
	    if ( KeyCode == Keys.ENTER )
	    {
		if ( MenuTop == 0 )
		{
		    remove( ScrollBar );
		    invalidate();
		}
		New_Game( true, false );
		return true;
	    }
	}

	// Return unsuccessfully, indicating that key was not handled.
	return false;
    }



    // ***** INTERNALLY-CALLED (PRIVATE) METHODS.



    // Displays the menu of available designs.

    private void Draw_Menu( boolean Invalidated )
    {
	int DesignNum, Index;

	// If required, determine which menu item should appear first.
	if ( Invalidated )
	{
	    MenuFirstDesign = CurrentDesign - ( MenuHeight / 2 );
	    if ( ( MenuFirstDesign + MenuHeight ) > NumDesigns )
	    {
		MenuFirstDesign = NumDesigns - MenuHeight;
	    }
	    if ( MenuFirstDesign < 0 )
	    {
		MenuFirstDesign = 0;
	    }
	    if ( MenuTop == 0 )
	    {
		ScrollBar.setValue( MenuFirstDesign );
	    }
	}

	// Colour the background.
	Display.setColor( Color.white );
	Display.fillRect( 0, 0, DisplayWidth, DisplayHeight );

	// Display all of the visible menu items.
	for ( Index = 0; Index < MenuHeight; Index++ )
	{
	    DesignNum = MenuFirstDesign + Index;
	    if ( DesignNum < NumDesigns && DesignNum != CurrentDesign )
	    {
		Draw_Menu_Item( MenuFirstDesign + Index, false );
	    }
	}

	// Highlight the current design last.
	Draw_Menu_Item( CurrentDesign, true );

	// Switch from menu mode to game mode.
	PlayingGame = false;
    }

    // Displays an individual menu item, highlighting if required.

    private void Draw_Menu_Item( int DesignNum, boolean DoHighlight )
    {
	String Title;
	int Top, Bottom, Width;

	// If the menu item is visible...
	Top = MenuTop + ( ( DesignNum - MenuFirstDesign ) * FontHeight );
	Bottom = Top + FontHeight - 1;
	if ( Top >= 0 && Bottom < DisplayHeight )
	{
	    // Highlight currently-selected design in different colour.
	    Title = (String) TitleList.elementAt( DesignNum );
	    Width = FontInfo.stringWidth( Title );
	    if ( DoHighlight && NumDesigns > 1 )
	    {
		Display.setColor( new Color(
			( 255 * DesignNum ) / ( NumDesigns - 1 ), 0,
			255 - ( ( 255 * DesignNum ) / ( NumDesigns - 1 ) ) ) );
	    }
	    else
	    {
		Display.setColor( Color.white );
	    }
	    Display.fillRect( 0, Top, DisplayWidth, FontHeight );

	    // Render the title, placing it in the centre.
	    if ( HaveCompleted[ DesignNum ] )
	    {
		Display.setColor( Color.green );
	    }
	    else if ( DoHighlight && NumDesigns > 1 )
	    {
		Display.setColor( Color.white );
	    }
	    else
	    {
		Display.setColor( Color.black );
	    }
	    Display.drawString( Title, ( DisplayWidth - Width ) / 2,
				Top + FontAscent );
	}
    }

    // Initialises the game with a specified design and displays it.

    private void New_Game( boolean NeedPanel, boolean IsRestart )
    {
	Vector DesignInfo;
	URL DataURL;
	int Row, Column;
	int Width, Index;
	char Type;

	// Add option panel if not already there.
	if ( NeedPanel )
	{
	    add( "South", Options );
	    validate();
	}

	// Store base URL for images.
	DataURL = getCodeBase();

	// Determine the basic height and width of the design.
	DesignInfo = (Vector) DesignList.elementAt( CurrentDesign );
	Rows = DesignInfo.size();
	Columns = 0;
	for ( Row = 0; Row < Rows; Row++ )
	{
	    Width = ( (String) DesignInfo.elementAt( Row ) ).length();
	    if ( Width > Columns )
	    {
		Columns = Width;
	    }
	}

	// Transfer the design definition into arena and locate player.
	NumUnmatched = 0;
	PlayerRow = -1;
	PlayerColumn = -1;
	Arena = new char[ Rows ][];
	for ( Row = 0; Row < Rows; Row++ )
	{
	    Arena[ Row ] = new char[ Columns ];
	    for ( Column = 0; Column < Columns; Column++ )
	    {
		if ( Column <
			( (String) DesignInfo.elementAt( Row ) ).length() )
		{
		    Type = ( (String) DesignInfo.elementAt( Row ) ).charAt(
								      Column );
		}
		else
		{
		    Type = '#';
		}
		Arena[ Row ][ Column ] = Type;
		if ( Type == '@' || Type == '+' )
		{
		    if ( PlayerRow < 0 )
		    {
			PlayerRow = Row;
			PlayerColumn = Column;
			if ( Type == '@' )
			{
			    Arena[ Row ][ Column ] = ' ';
			}
			else
			{
			    Arena[ Row ][ Column ] = '.';
			}
		    }
		    else
		    {
			if ( MultibanImage == null )
			{
			    MultibanImage =
				getImage( DataURL, "multiban.gif" );
			}
		    }
		}
		if ( Arena[ Row ][ Column ] == '.' ||
		     Arena[ Row ][ Column ] == '+' )
		{
		    NumUnmatched++;
		}
	    }
	}

	// Fail if the design doesn't contain a player/pusher character.
	if ( PlayerRow < 0 )
	{
	    ErrorMessage = "Design does not contain a player/pusher character.";
	    return;
	}

	// Begin a search - for walls only - from the current player location.
	Initialise_Search( PlayerRow, PlayerColumn, true );

	// Mark any unreachable floor cells as walls.
	for ( Row = 0; Row < Rows; Row++ )
	{
	    Column = First_Column( Row );
	    while ( Column < Columns )
	    {
		if ( Arena[ Row ][ Column ] == ' ' &&
		     ! Hop_Path( Row, Column ) )
		{
		    Arena[ Row ][ Column ] = '#';
		}
		Column = Next_Column( Column );
	    }
	}

	// If restarting, pop entries from undo list onto redo list...
	if ( IsRestart )
	{
	    while ( UndoList.size() > 0 )
	    {
		Index = UndoList.size() - 1;
		RedoList.addElement( (String) UndoList.elementAt( Index ) );
		UndoList.removeElementAt( Index );
	    }
	}
	// Otherwise, initialise movement histories used to undo/redo moves.
	else
	{
	    UndoList = new Vector();
	    RedoList = new Vector();
	}

	// If this is the first game, load all relevant images and determine
	// the sizes and offsets required to draw them.
	if ( IsFirstGame )
	{
	    DataURL = getCodeBase();
	    if ( IsHexoban )
	    {
		MarbleImage = getImage( DataURL, "marble2.jpg" );
		TargetImage = getImage( DataURL, "target2.jpg" );
		MatchImage = getImage( DataURL, "match2.jpg" );
		CellHeight = 46;
		CellWidth = 42;
		CellVerticalOffset = 35;
		MaskHeight = 210;
		MaskWidth = 210;
	    }
	    else
	    {
		MarbleImage = getImage( DataURL, "marble.jpg" );
		TargetImage = getImage( DataURL, "target.jpg" );
		MatchImage = getImage( DataURL, "match.jpg" );
		CellHeight = 40;
		CellWidth = 40;
		CellVerticalOffset = 40;
		MaskHeight = 240;
		MaskWidth = 240;
	    }
	    WoodImage = getImage( DataURL, "wood.jpg" );
	    PusherImage = getImage( DataURL, "ball.gif" );
	    BlobImage = getImage( DataURL, "blob.gif" );
	}

	// Calculate position of top-left cell so that design will be centred.
	GameTop = FontHeight +
		  ( ( Options.getBounds().y -
		    ( FontHeight + ( CellVerticalOffset * Rows ) +
		      ( CellHeight - CellVerticalOffset ) ) ) / 2 );
	if ( IsHexoban )
	{
	    GameLeft = ( DisplayWidth - ( CellWidth * ( Columns / 2 ) ) ) / 2;
	}
	else
	{
	    GameLeft = ( DisplayWidth - ( CellWidth * Columns ) ) / 2;
	}

	// Switch from menu mode to game mode.
	PlayingGame = true;

	// Display the new design.
	Draw_Arena();
    }

    // Displays the current design being played.

    private void Draw_Arena()
    {
	int Row, Column, Width;
	String Title;

	// Fill the background with pattern.
	Row = 0;
	while ( Row < DisplayHeight )
	{
	    Column = 0;
	    while ( Column < DisplayWidth )
	    {
		Display.drawImage( WoodImage, Column, Row, this );
		Column += 193;
	    }
	    Row += 201;
	}

	// Display the title of the current design.
	Title = (String) TitleList.elementAt( CurrentDesign );
	Width = FontInfo.stringWidth( Title );
	Display.setColor( Color.black );
	Display.drawString( Title, ( DisplayWidth - Width ) / 2, FontAscent );

	// Draw all of the cells that make up the design in its current state.
	for ( Row = 0; Row < Rows; Row++ )
	{
	    Column = First_Column( Row );
	    while ( Column < Columns )
	    {
		Draw_Cell( Row, Column );
		Column = Next_Column( Column );
	    }
	}
    }

    // Draws the contents of the specified cell.  Note: walls are pre-drawn
    // as a complete background, not as individual cells.

    private void Draw_Cell( int Row, int Column )
    {
	Image ImageType;
	int CellTop, CellLeft;
	int LineNum;
	int LineLeft, LineRight;
	boolean NeedExtension;

	// Determine main image to draw, return if there is nothing to draw.
	switch ( Arena[ Row ][ Column ] )
	{
	    case ' ' :
	    case '$' :
	    case '@' :
		ImageType = MarbleImage;
		break;
	    case '*' :
		ImageType = MatchImage;
		break;
	    case '.' :
	    case '+' :
		ImageType = TargetImage;
		break;
	    default :
		return;
	}

	// If playing Hexoban...
	if ( IsHexoban )
	{
	    // Locate the top-left corner of the cell.
	    CellTop = CellVerticalOffset * Row;
	    CellLeft = CellWidth * ( Column / 2 );
	    if ( Row % 2 == 1 )
	    {
		CellLeft += CellWidth / 2;
	    }

	    // Draw the segment of the image that corresponds to the hexagon
	    // shape line-by-line, wrapping at the right-hand edge of template.
	    for ( LineNum = 0; LineNum < 45; LineNum++ )
	    {
		LineLeft = CellLeft + HexagonInfo[ LineNum * 2 ];
		LineRight = CellLeft + HexagonInfo[ ( LineNum * 2 ) + 1 ];

		if ( LineRight % MaskWidth < LineLeft % MaskWidth )
		{
		    LineRight = ( ( CellLeft / MaskWidth ) * MaskWidth ) +
				( MaskWidth - 1 );
		    NeedExtension = true;
		}
		else
		{
		    NeedExtension = false;
		}

		Display.drawImage( ImageType, GameLeft + LineLeft,
				   GameTop + CellTop + LineNum,
				   GameLeft + LineRight + 1,
				   GameTop + CellTop + LineNum + 1,
				   LineLeft % MaskWidth,
				   ( CellTop + LineNum ) % MaskHeight,
				   ( LineRight % MaskWidth ) + 1,
				   ( ( CellTop + LineNum ) % MaskHeight ) + 1,
				   new Color( 0, 0, 0 ), this );

		if ( NeedExtension )
		{
		    LineLeft = ( ( CellLeft / MaskWidth ) + 1 ) * MaskWidth;
		    LineRight = CellLeft + HexagonInfo[ ( LineNum * 2 ) + 1 ];

		    Display.drawImage( ImageType, GameLeft + LineLeft,
				       GameTop + CellTop + LineNum,
				       GameLeft + LineRight + 1,
				       GameTop + CellTop + LineNum + 1,
				       LineLeft % MaskWidth,
				       ( CellTop + LineNum ) % MaskHeight,
				       ( LineRight % MaskWidth ) + 1,
				       ( ( CellTop + LineNum ) % MaskHeight ) +
					    1, new Color( 0, 0, 0 ), this );
		}
	    }
	}
	// Otherwise, copy section of image that corresponds to cell's position.
	else
	{
	    CellTop = CellHeight * Row;
	    CellLeft = CellWidth * Column;
	    Display.drawImage( ImageType,
			       GameLeft + CellLeft, GameTop + CellTop,
			       GameLeft + CellLeft + ( CellWidth - 1 ),
			       GameTop + CellTop + ( CellHeight - 1 ),
			       CellLeft % MaskWidth, CellTop % MaskHeight,
			       ( CellLeft % MaskWidth ) + ( CellWidth - 1 ),
			       ( CellTop % MaskHeight ) + ( CellHeight - 1 ),
			       new Color( 0, 0, 0 ), this );
	}

	// If the cell contains the player, draw on top of the cell image.
	if ( Row == PlayerRow && Column == PlayerColumn )
	{
	    if ( IsHexoban )
	    {
		Display.drawImage( PusherImage,
		       GameLeft + CellLeft + 6, GameTop + CellTop + 8, this );
	    }
	    else
	    {
		Display.drawImage( PusherImage,
		       GameLeft + CellLeft + 5, GameTop + CellTop + 4, this );
	    }
	}
	// Otherwise...
	{
	    // Use a displacement for Hexoban cells.
	    if ( IsHexoban )
	    {
		CellLeft++;
		CellTop += 3;
	    }

	    // If playing a Multiban design, draw any alternate player present.
	    if ( Arena[ Row ][ Column ] == '@' ||
		 Arena[ Row ][ Column ] == '+' )
	    {
		Display.drawImage( MultibanImage,
				GameLeft + CellLeft, GameTop + CellTop, this );
	    }

	    // If the cell contains a pushable box, draw on top of cell image.
	    if ( Arena[ Row ][ Column ] == '$' )
	    {
		Display.drawImage( BlobImage,
				GameLeft + CellLeft, GameTop + CellTop, this );
	    }
	}
    }

    // Attempts to move the player in a given direction and redraws affected
    // cells if successful.

    private void Make_Move( int RowOffset, int ColumnOffset, boolean KeepMove )
    {
	int OldRow, OldColumn;
	int NewRow, NewColumn;
	int FarRow, FarColumn;
	int PushCode;

	// Determine where the player would go.
	NewRow = PlayerRow + RowOffset;
	NewColumn = PlayerColumn + ColumnOffset;

	// If the destination is within the game area...
	if ( NewRow >= 0 && NewRow <= Rows &&
	     NewColumn >= 0 && NewColumn <= Columns )
	{
	    // Base action on contents of player's destination cell...
	    PushCode = 0;
	    switch ( Arena[ NewRow ][ NewColumn ] )
	    {
		// Return if the cell contains a wall.
		case '#' :
		    return;
		// Just allow the move if the cell is empty or a target.
		case ' ' :
		case '.' :
		    break;
		// If the cell contains another Multiban player, automatically
		// switch to using it instead of the current player and return.
		case '@' :
		case '+' :
		    Multiban_Move( NewRow, NewColumn, true );
		    return;
		// If the cell contains a box...
		default :
		    // Determine where the box would be pushed to.
		    FarRow = NewRow + RowOffset;
		    FarColumn = NewColumn + ColumnOffset;
		    // Try to move the box, return if something is in the way.
		    if ( Move_Box( NewRow, NewColumn, FarRow, FarColumn ) )
		    {
			return;
		    }
		    // Remember that a box was pushed.
		    PushCode = 13;
	    }

	    // Move the player and redraw in new location.
	    OldRow = PlayerRow;
	    OldColumn = PlayerColumn;
	    PlayerRow = -1;
	    PlayerColumn = -1;
	    Draw_Cell( OldRow, OldColumn );
	    PlayerRow = NewRow;
	    PlayerColumn = NewColumn;
	    Draw_Cell( PlayerRow, PlayerColumn );

	    // If this is a new move (not re-doing one previously undone),
	    // encode the move as a number and store it, indicating if a box
	    // was pushed, in the undo list.  Empty the redo list.
	    if ( KeepMove )
	    {
		UndoList.addElement( Integer.toString(
					( 5 * ( RowOffset + 1 ) ) +
					( ColumnOffset + 2 ) + PushCode ) );
		RedoList = new Vector();
	    }

	    // If the move completed the design, go on to the next one or
	    // return to the menu if the last design was completed.
	    if ( NumUnmatched == 0 )
	    {
		HaveCompleted[ CurrentDesign ] = true;
		if ( ( CurrentDesign + 1 ) == NumDesigns )
		{
		    remove( Options );
		    invalidate();
		    if ( MenuTop == 0 )
		    {
			add( "East", ScrollBar );
			validate();
		    }
		    Draw_Menu( true );
		}
		else
		{
		    CurrentDesign++;
		    New_Game( false, false );
		}
	    }
	}
    }

    // Changes which player/pusher is active when playing a Multiban design.

    private void Multiban_Move( int Row, int Column, boolean KeepMove )
    {
	int OldRow, OldColumn;

	// Transfer the active player/pusher into the current design state.
	if ( Arena[ PlayerRow ][ PlayerColumn ] == ' ' )
	{
	    Arena[ PlayerRow ][ PlayerColumn ] = '@';
	}
	else
	{
	    Arena[ PlayerRow ][ PlayerColumn ] = '+';
	}

	// Remove the selected player/pusher from the current design state.
	if ( Arena[ Row ][ Column ] == '@' )
	{
	    Arena[ Row ][ Column ] = ' ';
	}
	else
	{
	    Arena[ Row ][ Column ] = '.';
	}

	// Make the new player/pusher active.
	OldRow = PlayerRow;
	OldColumn = PlayerColumn;
	PlayerRow = Row;
	PlayerColumn = Column;

	// Redraw both affected cells.
	Draw_Cell( OldRow, OldColumn );
	Draw_Cell( Row, Column );

	// If this is a new move (not a undo or a redo), store the move in the
	// undo list and empty the redo list.
	if ( KeepMove )
	{
	    UndoList.addElement( OldRow + " " + OldColumn + " " +
				 Row + " " + Column );
	    RedoList = new Vector();
	}
    }

    // Moves the player/pusher back to their previous position, pulling any box
    // that was pushed to its original location.

    private void Undo_Move()
    {
	String Entry;
	StringTokenizer Tokeniser;
	int Index, Code, RowOffset, ColumnOffset;
	int OldRow, OldColumn;
	int NewRow, NewColumn;
	boolean Pushed;

	// If there is a move that can be undone...
	Index = UndoList.size() - 1;
	if ( Index >= 0 )
	{
	    // Fetch the last move and move it from undo list to redo list.
	    Entry = (String) UndoList.elementAt( Index );
	    RedoList.addElement( Entry );
	    UndoList.removeElementAt( Index );

	    // If the move was a 'hop' or a Multiban player/pusher switch...
	    if ( Entry.indexOf( " " ) >= 0 )
	    {
		// Determine which cell the player was in before the move.
		Tokeniser = new StringTokenizer( Entry );
		NewRow = Integer.parseInt( Tokeniser.nextToken() );
		NewColumn = Integer.parseInt( Tokeniser.nextToken() );

		// Move the player back to that cell.
		if ( Arena[ NewRow ][ NewColumn ] == ' ' ||
		     Arena[ NewRow ][ NewColumn ] == '.' )
		{
		    OldRow = PlayerRow;
		    OldColumn = PlayerColumn;
		    PlayerRow = NewRow;
		    PlayerColumn = NewColumn;
		    Draw_Cell( OldRow, OldColumn );
		    Draw_Cell( PlayerRow, PlayerColumn );
		}
		else
		{
		    Multiban_Move( NewRow, NewColumn, false );
		}
	    }
	    // Otherwise, if a normal Sokoban/Hexoban move...
	    else
	    {
		// Decode the move into offsets and push indicator.
		Code = Integer.parseInt( Entry );
		Pushed = ( Code > 13 );
		if ( Pushed )
		{
		    Code -= 13;
		}
		RowOffset = ( Code / 5 ) - 1;
		ColumnOffset = ( Code - ( 5 * ( RowOffset + 1 ) ) - 2 );

		// Move the player back to their original position and redraw.
		OldRow = PlayerRow;
		OldColumn = PlayerColumn;
		PlayerRow -= RowOffset;
		PlayerColumn -= ColumnOffset;
		Draw_Cell( PlayerRow, PlayerColumn );

		// If a box had been pushed, move it back to player's cell...
		if ( Pushed )
		{
		    NewRow = OldRow + RowOffset;
		    NewColumn = OldColumn + ColumnOffset;
		    Move_Box( NewRow, NewColumn, OldRow, OldColumn );
		    Draw_Cell( NewRow, NewColumn );
		}
		// Otherwise, redraw the cell that the player was in.
		else
		{
		    Draw_Cell( OldRow, OldColumn );
		}
	    }
	}
    }

    // Re-does a move that was previously undone.

    private void Redo_Move()
    {
	String Entry;
	StringTokenizer Tokeniser;
	int Index, Code;
	int Row, Column, OldRow, OldColumn;
	int RowOffset, ColumnOffset;

	// If there is a move that can be re-done...
	Index = RedoList.size() - 1;
	if ( Index >= 0 )
	{
	    // Fetch the last move and move it from redo list to undo list.
	    Entry = (String) RedoList.elementAt( Index );
	    UndoList.addElement( Entry );
	    RedoList.removeElementAt( Index );

	    // If the move was a 'hop' or a Multiban player/pusher switch...
	    if ( Entry.indexOf( " " ) >= 0 )
	    {
		// Determine which cell the player had moved to before the undo.
		Tokeniser = new StringTokenizer( Entry );
		Tokeniser.nextToken();
		Tokeniser.nextToken();
		Row = Integer.parseInt( Tokeniser.nextToken() );
		Column = Integer.parseInt( Tokeniser.nextToken() );

		// Move the player to that cell again.
		if ( Arena[ Row ][ Column ] == ' ' ||
		     Arena[ Row ][ Column ] == '.' )
		{
		    OldRow = PlayerRow;
		    OldColumn = PlayerColumn;
		    PlayerRow = Row;
		    PlayerColumn = Column;
		    Draw_Cell( OldRow, OldColumn );
		    Draw_Cell( PlayerRow, PlayerColumn );
		}
		else
		{
		    Multiban_Move( Row, Column, false );
		}
	    }
	    // Otherwise, if a normal Sokoban/Hexoban move...
	    else
	    {
		// Decode the move into offsets, ignoring push indicator.
		Code = Integer.parseInt( Entry );
		if ( Code > 13 )
		{
		    Code -= 13;
		}
		RowOffset = ( Code / 5 ) - 1;
		ColumnOffset = ( Code - ( 5 * ( RowOffset + 1 ) ) - 2 );

		// Redo the move as if the player were doing it, but don't
		// modify the undo and redo lists.
		Make_Move( RowOffset, ColumnOffset, false );
	    }
	}
    }

    // Moves a pushable box from one cell to another or returns 'true' if the
    // destination cell isn't empty.

    private boolean Move_Box( int OldRow, int OldColumn,
				int NewRow, int NewColumn )
    {
	// Base action on contents of box's destination cell...
	switch ( Arena[ NewRow ][ NewColumn ] )
	{
	    // Place the box in new cell if empty or a target and redraw it.
	    case ' ' :
	        Arena[ NewRow ][ NewColumn ] = '$';
	        Draw_Cell( NewRow, NewColumn );
	        break;
	    case '.' :
	        Arena[ NewRow ][ NewColumn ] = '*';
	        Draw_Cell( NewRow, NewColumn );
	        break;
	    // Return unsuccessfully if the destination cell isn't empty.
	    default :
	        return true;
	}

	// Remove the box from its old location.
	if ( Arena[ OldRow ][ OldColumn ] == '$' )
	{
	    Arena[ OldRow ][ OldColumn ] = ' ';
	}
	else
	{
	    Arena[ OldRow ][ OldColumn ] = '.';
	}

	// Update the unmatched target count.
	if ( Arena[ OldRow ][ OldColumn ] == ' ' &&
	     Arena[ NewRow ][ NewColumn ] == '*' )
	{
	    NumUnmatched--;
	}
	if ( Arena[ OldRow ][ OldColumn ] == '.' &&
	     Arena[ NewRow ][ NewColumn ] == '$' )
	{
	    NumUnmatched++;
	}

	// Return indicating that the box was moved.
	return false;
    }

    // Creates a search area, marking a specified starting cell and clearing
    // the distance values of all other cells.  Walls and cells containing
    // pushable boxes (or other Multiban player pieces) are marked as being
    // unavailable destinations.  If required, only cells containing walls are
    // considered obstructions.

    private void Initialise_Search( int StartRow, int StartColumn,
				    boolean AssumeEmpty )
    {
	int Row, Column;
	int Index;

	// Create a new search area.
	SearchArea = new int[ Rows ][ Columns ];

	// Determine which cells are empty or are targets and initialise them
	// as unsearched - mark others as unavailable destinations.
	for ( Row = 0; Row < Rows; Row++ )
	{
	    for ( Column = 0; Column < Columns; Column++ )
	    {
		if ( Arena[ Row ][ Column ] == ' ' ||
		     Arena[ Row ][ Column ] == '.' ||
		     ( AssumeEmpty && Arena[ Row ][ Column ] != '#' ) )
		{
		    SearchArea[ Row ][ Column ] = 0;
		}
		else
		{
		    SearchArea[ Row ][ Column ] = -1;
		}
	    }
	}

	// Start counting distance from starting cell.
	SearchArea[ StartRow ][ StartColumn ] = 1;
	SearchDistance = 2;
    }

    // Returns 'true' if there is a path from the player's current position to
    // the specified cell.

    private boolean Hop_Path( int HopRow, int HopColumn )
    {
	final int SokobanMoves[] = { -1,0, 1,0, 0,-1, 0,1 };
	final int HexobanMoves[] = { -1,-1, 0,-2, 1,-1, -1,1, 0,2, 1,1 };
	int Row, Column;
	int NewRow, NewColumn;
	int Index;
	boolean HasChanged;

	// Loop until a path has been found or search has been exhausted...
	HasChanged = true;
	while ( HasChanged && SearchArea[ HopRow ][ HopColumn ] == 0 )
	{
	    // Assume that all empty cells have been found by default.
	    HasChanged = false;

	    // For each cell...
	    for ( Row = 0; Row < Rows; Row++ )
	    {
		Column = First_Column( Row );
		while ( Column < Columns )
		{
		    // If the cell has the required count...
		    if ( SearchArea[ Row ][ Column ] == ( SearchDistance - 1 ) )
		    {
			// For each adjacent cell...
			if ( IsHexoban )
			{
			    Index = 10;
			}
			else
			{
			    Index = 6;
			}
			do
			{
			    if ( IsHexoban )
			    {
				NewRow = Row + HexobanMoves[ Index ];
				NewColumn = Column + HexobanMoves[ Index + 1 ];
			    }
			    else
			    {
				NewRow = Row + SokobanMoves[ Index ];
				NewColumn = Column + SokobanMoves[ Index + 1 ];
			    }

			    // If the cell is empty or is a target and it
			    // doesn't have a count, update it with the
			    // incremented distance.
			    if ( NewRow >= 0 && NewRow < Rows &&
				 NewColumn >= 0 && NewColumn < Columns &&
				 SearchArea[ NewRow ][ NewColumn ] == 0 )
			    {
				SearchArea[ NewRow ][ NewColumn ] =
								SearchDistance;
				HasChanged = true;
			    }

			    // Test the next adjacent cell.
			    Index -= 2;
			}
			while ( Index >= 0 );
		    }

		    // Advance to the next column.
		    Column = Next_Column( Column );
		}
	    }

	    // Advance to the next search distance.
	    SearchDistance++;
	}

	// Return indicating if there is a valid path.
	return SearchArea[ HopRow ][ HopColumn ] > 0;
    }

    // Returns 'true' if the pixel co-ordinates provided lie inside the
    // specified Hexoban cell.

    private boolean Inside_Hexoban_Cell( int CellRow, int CellColumn,
					 int PixelRow, int PixelColumn )
    {
	int CellTop, CellLeft;
	int LineNum;

	// Locate the top-left corner of the cell.
	CellTop = GameTop + ( CellVerticalOffset * CellRow );
	CellLeft = GameLeft + ( CellWidth * ( CellColumn / 2 ) );
	if ( CellRow % 2 == 1 )
	{
	    CellLeft += CellWidth / 2;
	}

	// If the pixel co-ordinates lie within the cell's bounding rectangle...
	if ( PixelRow >= CellTop &&
	     PixelRow < ( CellTop + CellHeight ) &&
	     PixelColumn >= CellLeft &&
	     PixelColumn < ( CellLeft + CellWidth ) )
	{
	    // For each horizontal pixel line defining the rectangle...
	    for ( LineNum = 0; LineNum < 45; LineNum++ )
	    {
		// Return successfully if the pixel co-ordinates lie
		// between the left and right edges of the hexagon's line.
		if ( PixelRow == CellTop + LineNum &&
		     PixelColumn >= CellLeft + HexagonInfo[ LineNum * 2 ] &&
		     PixelColumn <= CellLeft +
					HexagonInfo[ ( LineNum * 2 ) + 1 ] )
		{
		    return true;
		}
	    }
	}

	// Return indicating that no matching cell was found.
	return false;
    }

    // Returns the first column number that is valid for the specified row.

    private int First_Column( int Row )
    {
	// If playing Hexoban, return either 0 or 1 depending on the row.
	if ( IsHexoban )
	{
	    return Row % 2;
	}

	// Otherwise, simply return 0.
	return 0;
    }

    // Returns the next valid column number that follows the one specified.

    private int Next_Column( int Column )
    {
	// If playing Hexoban, skip unused column and return the one after.
	if ( IsHexoban )
	{
	    return Column + 2;
	}

	// Otherwise, simply return the next column.
	return Column + 1;
    }
}

